<?php

namespace App\Services;

use App\Models\BinaryWallet;
use App\Models\Configuration;
use App\Models\IncomeWallet;
use App\Models\Package;
use App\Models\Stockist;
use App\Models\UnilevelActivationUser;
use App\Models\UnilevelStat;
use App\Models\User;
use App\Models\UserPackageHistory;
use App\Models\UserSubscription;
use App\Models\UserWallet;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

use function Laravel\Prompts\select;

class CommissionService
{
    function __construct() {}

    function residualBonus($user)
    {
        CacheService::removeAll();
        $config = CacheService::configuration();
        $welBonusObj = json_decode($config->residual_ref_bonus);
        $levelDeepCounts = count($welBonusObj);

        $uplines = User::uplineByReferral($user->id, $levelDeepCounts);

        foreach ($uplines as $index => $value) {
            $bonus = ((float) $config->registration_fee * (int) $welBonusObj[$index]->bonus) / 100;

            if ($index == 0) {
                $remark = 'Residual bonus from ' . ucfirst($user->username);
            } else {
                $remark = 'Level-' . ($index + 1) . ' residual bonus from ' . ucfirst($user->username);
            }

            if ($value and $value->id && $bonus) {
                IncomeWallet::create([
                    'user_id' => $value->id,
                    'type' => 'residual',
                    'amount' => $bonus,
                    'remark' => $remark,
                ]);

                $userWallet = UserWallet::where('user_id', $value->id)->first();

                if ($userWallet) {
                    $userWallet->master_wallet += $bonus;
                    $userWallet->residual_b += $bonus;
                    $userWallet->save();
                } else {
                    UserWallet::create([
                        'user_id' => $value->id,
                        'master_wallet' => $bonus,
                        'residual_b' => $bonus,
                    ]);
                }
            }

            set_time_limit(60 * 60);
        }

        return true;
    }

    function welcomeBonus($user)
    {
        CacheService::removeAll();
        $config = CacheService::configuration();
        $welBonusObj = json_decode($config->welcome_ref_bonus);
        $levelDeepCounts = count($welBonusObj);

        $uplines = User::uplineByReferral($user->id, $levelDeepCounts);

        foreach ($uplines as $index => $value) {
            $bonus = ((float) $config->registration_fee * (int) $welBonusObj[$index]->bonus) / 100;

            if ($index == 0) {
                $remark = 'Welcome bonus from ' . ucfirst($user->username);
            } else {
                $remark = 'Level-' . ($index + 1) . ' welcome bonus from ' . ucfirst($user->username);
            }

            if ($value and $value->id && $bonus) {
                IncomeWallet::create([
                    'user_id' => $value->id,
                    'type' => 'welcome',
                    'amount' => $bonus,
                    'remark' => $remark,
                ]);

                $userWallet = UserWallet::where('user_id', $value->id)->first();

                if ($userWallet) {
                    $userWallet->master_wallet += $bonus;
                    $userWallet->welcome_b += $bonus;
                    $userWallet->save();
                } else {
                    UserWallet::create([
                        'user_id' => $value->id,
                        'master_wallet' => $bonus,
                        'welcome_b' => $bonus,
                    ]);
                }
            }

            set_time_limit(60 * 60);
        }

        return true;
    }

    function referralBonus($user, $bv, $type)
    {
        //return $user->username;
        if (strtolower(trim($type)) == 'reorder') return false;

        $single = UserSubscription::where('user_id', $user->id)->first();
        $maxPackageBV = Package::max('min_pv');

        if ($single->bv == $maxPackageBV) {
            $subHistory = UserPackageHistory::latest('id')->where('user_id',$user->id)->first();
            $excessBV = ((($single->bv - $subHistory->bv) + $bv) - $maxPackageBV);
            
            return self::salesBonus($user, $excessBV, 'reorder');
        }

        //CacheService::removeAll();
        $config = Configuration::first();
        $uplines = User::uplines($user->username, $config->unilevel_depth);
        for ($i = 0; $i < $config->unilevel_depth; $i++) {
            $level = $i + 1;
            if (isset($uplines[$i])) {
                $upline = $uplines[$i];
                $subscription = UserSubscription::where('user_id', $upline->id)->with('package')->first();
                if (isset($subscription->package) && $subscription->package->referral_perc) {
                    $percs = explode('|', $subscription->package->referral_perc);
                    if (isset($percs[$i])) {
                        $perc = (int) $percs[$i];

                        $bonus = (($perc * $bv) / 100) * $config->usd_rate;

                        $remark = 'Referral commission from level-' . $level . ' ' . ucfirst($type);
                        if ($i == 0) {
                            $remark = 'Referral commission from ' . ucfirst($user->username) . ' ' . ucfirst($type);
                        }

                        IncomeWallet::create([
                            'user_id' => $upline->id,
                            'type' => 'referral',
                            'amount' => $bonus,
                            'e_points' => $bv,
                            'remark' => $remark,
                        ]);

                        $refWallet = UserWallet::where('user_id', $upline->id)->first();
                        if ($refWallet) {
                            $refWallet->master_wallet += $bonus;
                            $refWallet->referral_b += $bonus;
                            $refWallet->save();
                        } else {
                            UserWallet::create([
                                'user_id' => $upline->id,
                                'master_wallet' => $bonus,
                                'referral_b' => $bonus
                            ]);
                        }
                    }
                }
            }
        }

        return false;
    }

    function salesBonus($user, $bv, $type)
    {
        // return $user->username;
        if (strtolower(trim($type)) != 'reorder') return false;

        $config = Configuration::first();
        $uplines = User::uplines($user->username, $config->unilevel_depth);

        // Add the buyer in front of the uplines array
        array_unshift($uplines, $user);

        for ($i = 0; $i < count($uplines); $i++) {
            $level = $i + 1;
            
            if (isset($uplines[$i])) {
                $upline = $uplines[$i];
                $subscription = UserSubscription::where('user_id', $upline->id)->with('package')->first();
                if (isset($subscription->package) && $subscription->package->sales_perc) {
                    $percs = explode('|', $subscription->package->sales_perc);
                    if (isset($percs[$i])) {
                        $perc = (int) $percs[$i];

                        $bonus = (($perc * $bv) / 100) * $config->usd_rate;

                        $remark = 'Indirect Sales Bonus from level-' . $level . ' ' . ucfirst($type);
                        $incomeType = 'indirect_sales';                        
                        $walletTypeColumn = 'indirect_sales_b';
                        if ($i == 0) {
                            $incomeType = 'direct_sales';
                            $walletTypeColumn = 'direct_sales_b';
                            $remark = ucfirst($type) .' Direct Sales Bonus' ;
                        }

                        IncomeWallet::create([
                            'user_id' => $upline->id,
                            'type' => $incomeType,
                            'amount' => $bonus,
                            'e_points' => $bv,
                            'remark' => $remark,
                        ]);

                        $refWallet = UserWallet::where('user_id', $upline->id)->first();
                        if ($refWallet) {
                            $refWallet->master_wallet += $bonus;
                            $refWallet->$walletTypeColumn += $bonus;
                            $refWallet->save();
                        } else {
                            UserWallet::create([
                                'user_id' => $upline->id,
                                'master_wallet' => $bonus,
                                $walletTypeColumn => $bonus
                            ]);
                        }
                    }
                }
            }
        }

        return false;
    }


    // Create matching bonus if right_pv and left_pv are 100 or "binary_min_point" and above
    function binaryBonus($take = 50)
    {
        CacheService::removeAll();
        $config = Configuration::first();
        $minBV = $config->binary_min_point;
        $inQuery = BinaryWallet::select(
            'binary_wallets.*',
            'packages.binary_perc',
            'packages.daily_match',
            'packages.id as package_id'
        )
            ->whereRaw("
                left_bv > 0 AND left_bv >= $config->binary_min_point AND
                right_bv > 0 AND right_bv >= $config->binary_min_point
            ")
            ->join('user_subscriptions', 'user_subscriptions.user_id', 'binary_wallets.user_id')
            ->join('packages', 'packages.id', 'user_subscriptions.package_id');

        $timesRun = ceil($inQuery->count() / $take);
        $ran = 0;
        while ($ran <= $timesRun) {
            $offset = ($take * $ran);
            $rows = $inQuery->take($take)->offset($offset)->get();
            foreach ($rows as $row) {
                $left = $row->left_bv;
                $right = $row->right_bv;

                if ($minBV == 0) {
                    $minBV = ($left > $right) ? $right : $left;
                }

                $forToday = IncomeWallet::where(['user_id' => $row->user_id, 'type' => 'binary'])
                    ->where('created_at', 'like', date("Y-m-d") . "%")
                    ->count();

                // Create matching bonus
                while (($left >= $minBV && $right >= $minBV) && $row->daily_match > $forToday) {
                    if ($row->binary_perc) {
                        $bonusUSD = ($row->binary_perc * $minBV) / 100;
                        $bonus = $bonusUSD * $config->usd_rate;
                        IncomeWallet::create([
                            'user_id' => $row->user_id,
                            'type' => 'binary',
                            'amount' => $bonus,
                            'e_points' => $minBV,
                            'remark' => 'Binary match bonus',
                        ]);

                        $wallet = UserWallet::where('user_id', $row->user_id)->first();

                        if ($wallet) {
                            $wallet->master_wallet += $bonus;
                            $wallet->binary_b += $bonus;
                            $wallet->save();
                        } else {
                            UserWallet::create([
                                'user_id' => $row->user_id,
                                'master_wallet' => $bonus,
                                'binary_b' => $bonus,
                            ]);
                        }

                        $this->leverageBonus($row, $bonusUSD);

                        $left -= $minBV;
                        $right -= $minBV;
                        $forToday += 1;
                    }
                }

                // Update PVs after creating matching bonus
                BinaryWallet::where('user_id', $row->user_id)->update([
                    'left_bv' => $left,
                    'right_bv' => $right,
                ]);
            }

            set_time_limit(60 * 60);
            $ran++;
        }

        return true;
    }

    function unilevelActivationBonus($user, $pv, $type)
    {
        if (strtolower(trim($type)) == 'reorder') return false;

        $ref = User::select('id', 'username')->where('username', $user->referral)->first();
        if (!$ref) return false;

        $unilevelMatch = UnilevelActivationUser::where([
            'user_id' => $user->id,
            'sponsor_id' => $ref->id,
        ])->first();

        if (!$unilevelMatch) return false;

        CacheService::removeAll();
        $config = CacheService::configuration();
        $bonus = (($config->unilevel_activation_bonus * $pv) / 100) * $config->usd_rate;

        IncomeWallet::create([
            'user_id' => $unilevelMatch->sponsor_id,
            'type' => 'unilevel_activation',
            'amount' => $bonus,
            'remark' => 'Unilevel Alternative Activation bonus from ' . ucfirst($user->username),
        ]);

        $userWallet = UserWallet::where('user_id', $unilevelMatch->sponsor_id)->first();
        if ($userWallet) {
            $userWallet->master_wallet += $bonus;
            $userWallet->unilevel_activation_b += $bonus;
            $userWallet->save();
        } else {
            UserWallet::create([
                'user_id' => $unilevelMatch->sponsor_id,
                'master_wallet' => $bonus,
                'unilevel_activation_b' => $bonus,
            ]);
        }

        return true;
    }

    function leverageBonus($binaryInfo, $binaryEarnedRawPoint)
    {
        $secondHighestPackage = Package::latest('id')->skip(1)->first();
        $user = User::select('id', 'username')->find($binaryInfo->user_id);
        if (!$user) return false;

        CacheService::removeAll();
        $config = CacheService::configuration();
        $uplines = User::uplinesByPlacement($user->username, 5);

        $level = 1;
        foreach ($uplines as $upline) {
            $sub = UserSubscription::where('user_id', $upline->id)->first();
            $binaryChecker = User::binaryChecker($upline->username);
            $autoshipChecker = OrderService::autoshipChecker($upline->id);

            if ($sub && $sub->package_id) {
                $bonus = 0;
                $rawPoints = ($config->leverage_bonus * $binaryEarnedRawPoint) / 100;
                if ($level <= 4) {
                    $bonus = $rawPoints * $config->usd_rate;
                } else if ($level > 4 && $sub->package_id >= $secondHighestPackage->id) {
                    $bonus = $rawPoints * $config->usd_rate;
                }

                if ($bonus > 0) {
                    DB::beginTransaction();
                    try {
                        $remark = '';
                        if (!$autoshipChecker && !$binaryChecker) {
                            $remark = 'Lost Team bonus - Reasons: Incomplete binary & no autoship, Level-' . $level . ', ' . ucfirst($upline->username);
                        } else if (!$autoshipChecker && $binaryChecker) {
                            $remark = 'Lost Team bonus - Reasons: No autoship, Level-' . $level . ', ' . ucfirst($upline->username);
                        } else if ($autoshipChecker && !$binaryChecker) {
                            $remark = 'Lost Team bonus - Reasons: Incomplete binary, Level-' . $level . ', ' . ucfirst($upline->username);
                        }

                        if ($autoshipChecker && $binaryChecker) {
                            $remark = 'Level ' . $level . ' Team bonus from ' . ucfirst($upline->username);
                            $wallet = UserWallet::where('user_id', $upline->id)->first();
                            if ($wallet) {
                                $wallet->master_wallet += $bonus;
                                $wallet->team_b += $bonus;
                                $wallet->save();
                            } else {
                                UserWallet::create([
                                    'user_id' => $upline->id,
                                    'master_wallet' => $bonus,
                                    'team_b' => $bonus,
                                ]);
                            }
                        }

                        IncomeWallet::create([
                            'user_id' => $upline->id,
                            'type' => 'team_bonus',
                            'amount' => $bonus,
                            'remark' => $remark,
                        ]);
                        DB::commit();
                    } catch (\Exception $e) {
                        DB::rollBack();
                    }
                }
            }

            set_time_limit(60 * 60);
            $level++;
            if ($level >= 6) return true;
        }

        return true;
    }
}
